const OFFSCREEN_URL = "offscreen.html";
const PROFILE_URL = "https://throne.com/aira_real";
const RELOAD_DELAY_MS = 30000;
const RELOAD_PREFIX = "aira_reload:";
const GUARD_PREFIX = "aira_reload_guard:";

async function ensureOffscreen() {
  if (chrome.offscreen && chrome.offscreen.hasDocument) {
    try { if (await chrome.offscreen.hasDocument()) return; } catch (e) {}
  }
  try {
    await chrome.offscreen.createDocument({
      url: OFFSCREEN_URL,
      reasons: ["AUDIO_PLAYBACK"],
      justification: "Play tier roulette audio"
    });
  } catch (e) {}
}

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg && msg.type === "aira_play_audio") {
    ensureOffscreen().then(() => {
      chrome.runtime.sendMessage({ type: "aira_offscreen_play" });
    });
    sendResponse({ ok: true });
    return true;
  }
});

function guardKey(tabId) { return GUARD_PREFIX + tabId; }

async function getGuard(tabId) {
  const k = guardKey(tabId);
  const o = await chrome.storage.session.get(k);
  return o[k] || 0;
}

async function setGuard(tabId, ts) {
  const k = guardKey(tabId);
  const o = {};
  o[k] = ts;
  await chrome.storage.session.set(o);
}

async function clearGuard(tabId) {
  const k = guardKey(tabId);
  await chrome.storage.session.remove(k);
}

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo) => {
  if (!changeInfo.url) return;
  if (!/^https:\/\/throne\.com\/aira_real\/checkout\/[^?#]*/i.test(changeInfo.url)) return;
  const now = Date.now();
  const guard = await getGuard(tabId);
  if (guard && guard > now) return;
  await setGuard(tabId, now + RELOAD_DELAY_MS + 5000);
  chrome.alarms.create(RELOAD_PREFIX + tabId, { when: now + RELOAD_DELAY_MS });
});

chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (!alarm || !alarm.name || alarm.name.indexOf(RELOAD_PREFIX) !== 0) return;
  const tabId = parseInt(alarm.name.slice(RELOAD_PREFIX.length), 10);
  try {
    await chrome.tabs.update(tabId, { url: PROFILE_URL });
  } catch (e) {}
  await clearGuard(tabId);
});
